<?php
/**
 * Plugin Name: SpendCheck
 * Description: Invoice management and spending analysis
 * Version: 1.0
 * Author: SpendCheck
 */

if (!defined('ABSPATH')) {
    exit;
}

// Create tables on plugin activation
function spendcheck_activate() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    // Companies table
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}spendcheck_companies (
        id varchar(36) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        user_id bigint(20) unsigned NOT NULL,
        company_name varchar(255) NOT NULL,
        post_code varchar(20) NOT NULL,
        business_type varchar(50) NOT NULL,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        FOREIGN KEY (user_id) REFERENCES {$wpdb->users}(ID) ON DELETE CASCADE
    ) $charset_collate;";
    dbDelta($sql);

    // Suppliers table
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}spendcheck_suppliers (
        id varchar(36) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        name varchar(255) NOT NULL,
        supplier_code varchar(50),
        PRIMARY KEY (id)
    ) $charset_collate;";
    dbDelta($sql);

    // Invoices table
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}spendcheck_invoices (
        id varchar(36) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        company_id varchar(36) NOT NULL,
        supplier_id varchar(36) NOT NULL,
        invoice_number varchar(50) NOT NULL,
        invoice_date date NOT NULL,
        supplier_order_number varchar(50),
        customer_po_number varchar(50),
        scan_date datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY company_id (company_id),
        KEY supplier_id (supplier_id),
        FOREIGN KEY (company_id) REFERENCES {$wpdb->prefix}spendcheck_companies(id) ON DELETE CASCADE,
        FOREIGN KEY (supplier_id) REFERENCES {$wpdb->prefix}spendcheck_suppliers(id) ON DELETE CASCADE
    ) $charset_collate;";
    dbDelta($sql);

    // Invoice items table
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}spendcheck_invoice_items (
        id varchar(36) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        invoice_id varchar(36) NOT NULL,
        product_code varchar(50) NOT NULL,
        product_description text NOT NULL,
        unit_price decimal(10,2) NOT NULL,
        quantity decimal(10,2) NOT NULL,
        product_match_id varchar(36),
        PRIMARY KEY (id),
        KEY invoice_id (invoice_id),
        FOREIGN KEY (invoice_id) REFERENCES {$wpdb->prefix}spendcheck_invoices(id) ON DELETE CASCADE
    ) $charset_collate;";
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'spendcheck_activate');

// REST API endpoints
function spendcheck_register_routes() {
    register_rest_route('spendcheck/v1', '/company', array(
        'methods' => 'POST',
        'callback' => 'spendcheck_create_company',
        'permission_callback' => function() {
            return is_user_logged_in();
        }
    ));

    register_rest_route('spendcheck/v1', '/invoice', array(
        'methods' => 'POST',
        'callback' => 'spendcheck_create_invoice',
        'permission_callback' => function() {
            return is_user_logged_in();
        }
    ));

    register_rest_route('spendcheck/v1', '/invoices', array(
        'methods' => 'GET',
        'callback' => 'spendcheck_get_invoices',
        'permission_callback' => function() {
            return is_user_logged_in();
        }
    ));
}
add_action('rest_api_init', 'spendcheck_register_routes');

// API Handlers
function spendcheck_create_company($request) {
    global $wpdb;
    $user_id = get_current_user_id();
    $params = $request->get_params();

    $wpdb->insert(
        $wpdb->prefix . 'spendcheck_companies',
        array(
            'id' => wp_generate_uuid4(),
            'user_id' => $user_id,
            'company_name' => $params['company_name'],
            'post_code' => $params['post_code'],
            'business_type' => $params['business_type']
        )
    );

    return new WP_REST_Response(array('status' => 'success'), 200);
}

function spendcheck_create_invoice($request) {
    global $wpdb;
    $params = $request->get_params();
    
    // Start transaction
    $wpdb->query('START TRANSACTION');
    
    try {
        // Create or get supplier
        $supplier = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}spendcheck_suppliers WHERE name = %s",
            $params['supplier_name']
        ));

        $supplier_id = $supplier ? $supplier->id : wp_generate_uuid4();
        
        if (!$supplier) {
            $wpdb->insert(
                $wpdb->prefix . 'spendcheck_suppliers',
                array(
                    'id' => $supplier_id,
                    'name' => $params['supplier_name']
                )
            );
        }

        // Create invoice
        $invoice_id = wp_generate_uuid4();
        $wpdb->insert(
            $wpdb->prefix . 'spendcheck_invoices',
            array(
                'id' => $invoice_id,
                'company_id' => $params['company_id'],
                'supplier_id' => $supplier_id,
                'invoice_number' => $params['invoice_number'],
                'invoice_date' => $params['invoice_date'],
                'supplier_order_number' => $params['supplier_order_number'],
                'customer_po_number' => $params['customer_po_number']
            )
        );

        // Create invoice items
        foreach ($params['items'] as $item) {
            $wpdb->insert(
                $wpdb->prefix . 'spendcheck_invoice_items',
                array(
                    'id' => wp_generate_uuid4(),
                    'invoice_id' => $invoice_id,
                    'product_code' => $item['product_code'],
                    'product_description' => $item['description'],
                    'unit_price' => $item['unit_price'],
                    'quantity' => $item['quantity']
                )
            );
        }

        $wpdb->query('COMMIT');
        return new WP_REST_Response(array('status' => 'success'), 200);
    } catch (Exception $e) {
        $wpdb->query('ROLLBACK');
        return new WP_Error('invoice_error', $e->getMessage(), array('status' => 500));
    }
}

function spendcheck_get_invoices($request) {
    global $wpdb;
    $user_id = get_current_user_id();

    $invoices = $wpdb->get_results($wpdb->prepare(
        "SELECT i.*, s.name as supplier_name
         FROM {$wpdb->prefix}spendcheck_invoices i
         JOIN {$wpdb->prefix}spendcheck_suppliers s ON i.supplier_id = s.id
         JOIN {$wpdb->prefix}spendcheck_companies c ON i.company_id = c.id
         WHERE c.user_id = %d
         ORDER BY i.created_at DESC",
        $user_id
    ));

    return new WP_REST_Response($invoices, 200);
}