import React, { useCallback, useState } from 'react';
import { useDropzone } from 'react-dropzone';
import { createWorker } from 'tesseract.js';
import { Upload, Loader2 } from 'lucide-react';
import { extractInvoiceData, ExtractedInvoiceData } from '../services/ollama';

interface InvoiceUploadProps {
  onDataExtracted: (data: ExtractedInvoiceData) => void;
  onError: (error: string) => void;
}

export default function InvoiceUpload({ onDataExtracted, onError }: InvoiceUploadProps) {
  const [isProcessing, setIsProcessing] = useState(false);

  const processFile = async (file: File) => {
    try {
      setIsProcessing(true);

      // Initialize Tesseract worker
      const worker = await createWorker();
      await worker.loadLanguage('eng');
      await worker.initialize('eng');

      // Read the image
      const { data: { text } } = await worker.recognize(file);
      await worker.terminate();

      // Extract data using Ollama
      const extractedData = await extractInvoiceData(text);
      onDataExtracted(extractedData);
    } catch (error) {
      console.error('Error processing invoice:', error);
      onError(error instanceof Error ? error.message : 'Failed to process invoice');
    } finally {
      setIsProcessing(false);
    }
  };

  const onDrop = useCallback((acceptedFiles: File[]) => {
    if (acceptedFiles.length > 0) {
      processFile(acceptedFiles[0]);
    }
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.bmp'],
      'application/pdf': ['.pdf'],
    },
    maxFiles: 1,
  });

  return (
    <div
      {...getRootProps()}
      className={`p-8 border-2 border-dashed rounded-lg text-center cursor-pointer transition-colors
        ${isDragActive ? 'border-blue-500 bg-blue-50' : 'border-gray-300 hover:border-blue-500'}`}
    >
      <input {...getInputProps()} />
      <div className="space-y-4">
        {isProcessing ? (
          <>
            <Loader2 className="h-10 w-10 mx-auto text-blue-500 animate-spin" />
            <p className="text-gray-600">Processing invoice...</p>
          </>
        ) : (
          <>
            <Upload className="h-10 w-10 mx-auto text-gray-400" />
            <div>
              <p className="text-gray-600">
                {isDragActive
                  ? 'Drop the invoice here'
                  : 'Drag and drop an invoice, or click to select'}
              </p>
              <p className="text-sm text-gray-500 mt-1">
                Supports PNG, JPG, GIF, BMP and PDF
              </p>
            </div>
          </>
        )}
      </div>
    </div>
  );
}